// $Id: CMenuWindow.cpp,v 1.10 2007/02/27 19:47:11 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

//	===========================================================================

#include <Host/CDisplayInformation.hpp>
#include "CMenuWindow.hpp"
#include "../Controls/CMenuItemDisplay.hpp"
#include "../Controls/CHtmlMenuItem.hpp"
#include "../Controls/CSelectableMenuItem.hpp"

//	===========================================================================

using Exponent::Host::CDisplayInformation;
using Exponent::GUI::Windowing::CMenuWindow;
using Exponent::GUI::Controls::CMenuItemDisplay;
using Exponent::GUI::Controls::CHtmlMenuItem;
using Exponent::GUI::Controls::CSelectableMenuItem;

#ifdef WIN32
#include <gdiplus.h>
using namespace Gdiplus;
#endif

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CMenuWindow, CWindow);

//	===========================================================================
CMenuWindow::CMenuWindow(const CRect &size, CWindowAttributes *parentAttributes, CMenu *menu, const bool useIcons, IMenuListener *listener) 
		   : CWindow(size)
		   , m_menuListener(NULL)
		   , m_mySizeOnInitialise(NULL)
		   , m_subWindow(NULL)
		   , m_parentMenuWindow(NULL)
		   , m_openingSubMenuWindow(false)
		   , m_useIcons(true)
#ifndef WIN32
		   , m_closingWindow(false)
		   , m_windowGroup(NULL)
#endif
{
	EXPONENT_CLASS_CONSTRUCTION(CMenuWindow);

#ifndef WIN32
	CreateWindowGroup(kWindowGroupAttrSelectAsLayer    |
                      kWindowGroupAttrMoveTogether     |
					  kWindowGroupAttrLayerTogether	   |
                      kWindowGroupAttrSharedActivation |
                      kWindowGroupAttrHideOnCollapse,
					  &m_windowGroup);
#endif

	// Null the windows
	NULL_POINTER(m_subWindow);
	NULL_POINTER(m_parentMenuWindow);

	// Copy the attributes
	char title[256];
	sprintf(title, "Menu_%li_%li", (long)&(useIcons), (long)&(size));

	*m_attributes = *parentAttributes;
	m_attributes->setWindowTitle(title);
	this->setWindowAlpha(m_attributes->getWindowAlphaValue());

	// Initialise
#ifdef WIN32
	m_attributes->initialise(WS_POPUP, WS_EX_LAYERED | WS_EX_NOACTIVATE | WS_EX_TOPMOST);
#else  
	m_attributes->initialise(kWindowNoAttributes | kWindowStandardHandlerAttribute | kWindowNoShadowAttribute, kPlainWindowClass);
	m_attributes->windowHasShadow(false);
	m_closingWindow = false;
#endif

	m_openingSubMenuWindow = false;
	m_useIcons			   = useIcons;

	// Delete the old control root and replace with the new one
	FREE_POINTER(m_theControlRoot);
	m_theControlRoot = new CMenuRoot(this);
	((CMenuRoot *)m_theControlRoot)->setRootArea(size);
	((CMenuRoot *)m_theControlRoot)->useIcons(m_useIcons);
	this->setMinimumWidth(size.getWidth());
	
	// Reregister the window
	this->registerDropFileListener(NULL);
	this->registerKeyboardListener(NULL);
	this->registerMouseListener(m_theControlRoot);
	this->registerMenuListener(listener);

	// NULL the pointer
	NULL_POINTER(m_mySizeOnInitialise);

	// Store the menu
	this->setMenu(menu);

	// Initialise the window
	this->initialiseWindow(m_attributes, false);
	
#ifndef WIN32
#endif
}

//	===========================================================================
CMenuWindow::~CMenuWindow()
{
	EXPONENT_CLASS_DESTRUCTION(CMenuWindow);
	FREE_POINTER(m_mySizeOnInitialise);
	FREE_POINTER(m_subWindow);
	NULL_POINTER(m_parentMenuWindow);
#ifndef WIN32
	ReleaseWindowGroup (m_windowGroup);
#endif
}

//	===========================================================================
void CMenuWindow::setMinimumWidth(const long width)
{
	((CMenuRoot *)m_theControlRoot)->setMinimumWidth(width);
}

//	===========================================================================
void CMenuWindow::openPopupWindow(const CPoint &point)
{
	this->setWindowPosition(point);
	if (m_mySizeOnInitialise)
	{
		this->setWindowSize(*m_mySizeOnInitialise);
		((CMenuRoot *)m_theControlRoot)->setRootArea(CRect(0, 0, m_mySizeOnInitialise->getWidth(), m_mySizeOnInitialise->getHeight()));

		// Shift so that we are always visible...
		const long height    = CDisplayInformation::getMonitorScreenHeight();
		const long width     = CDisplayInformation::getMonitorScreenWidth();
		const long ourHeight = m_mySizeOnInitialise->getHeight();
		const long ourWidth  = m_mySizeOnInitialise->getWidth();
		long yPos		     = point.getYPosition();
		long xPos		     = point.getXPosition();
		while (yPos + ourHeight > height && yPos > 0)
		{
			yPos--;
		}
		while (xPos + ourWidth > width && xPos > 0)
		{
			xPos--;
		}

		this->setWindowPosition(CPoint(xPos, yPos));
	}
	this->openWindow();
	if (m_menuListener)
	{
		m_menuListener->handleMenuOpening(CMenuEvent(NULL, ((CMenuRoot *)m_theControlRoot)->getMenu()));
	}
#ifndef WIN32
	// Set the window group to be that of a floating window, so that Application which break the
	// Aqua L+F and use floating windows will still be able to see this window at the top of the z ordering tree...
	// Add it to the top level window group
	SetWindowGroup(this->getMutableWindowHandle()->m_windowHandle, m_windowGroup);
	SetWindowGroupLevel(m_windowGroup, kCGMainMenuWindowLevel); 
	//SetWindowActivationScope(this->getMutableWindowHandle()->m_windowHandle, kWindowActivationScopeAll);
	CWindowTools::setFocusWindow(this->getMutableWindowHandle());
	this->updateWindow();
#endif
}

//	===========================================================================
void CMenuWindow::closePopupWindow()
{
	FREE_POINTER(m_subWindow);
	this->closeWindow();
	if (m_menuListener)
	{
		m_menuListener->handleMenuClosing(CMenuEvent(NULL, ((CMenuRoot *)m_theControlRoot)->getMenu()));
	}
}

//	===========================================================================
void CMenuWindow::gainedFocus(IWindow *window)
{
	// Nothing just yet
}

//	===========================================================================
void CMenuWindow::lostFocus(IWindow *window)
{
#ifdef WIN32
	if (!m_openingSubMenuWindow)
	{
		this->closePopupWindow();
		if (m_parentMenuWindow)
		{
			m_parentMenuWindow->lostFocus(this);
		}
	}
#else
	if (!m_openingSubMenuWindow && !m_closingWindow)
	{
		if (m_parentMenuWindow)
		{
			m_parentMenuWindow->lostFocus(this);
		}
		else
		{
			this->closeWindow();
			if (m_menuListener)
			{
				m_menuListener->handleMenuClosing(CMenuEvent(NULL, ((CMenuRoot *)m_theControlRoot)->getMenu()));
			}
		}
		if (m_subWindow)
		{
			m_subWindow->closeWindow();
		}
	}
#endif
}	

//	===========================================================================
void CMenuWindow::setMenu(CMenu *menu)
{
	if (menu)
	{
		m_mySizeOnInitialise = ((CMenuRoot *)m_theControlRoot)->setMenu(menu);
	}
}

//	===========================================================================
void CMenuWindow::setMenuAndSize(CMenu *menu)
{
	this->setMenu(menu);
	this->setWindowSize(*m_mySizeOnInitialise);
}

//	===========================================================================
void CMenuWindow::setFont(CFont *font)
{
	((CMenuRoot *)m_theControlRoot)->setFont(font);
}

//	===========================================================================
void CMenuWindow::setMenuColours(const SMenuColours &colours)
{
	((CMenuRoot *)m_theControlRoot)->setMenuColours(colours);
}

//	===========================================================================
void CMenuWindow::setCurrentSelection(const long selection)
{
	((CMenuRoot *)m_theControlRoot)->setCurrentSelection(selection);
}

//	===========================================================================
void CMenuWindow::handleMenuOpening(const CMenuEvent &event)
{
	// Dont need to do anything
}

//	===========================================================================
void CMenuWindow::handleMenuClosing(const CMenuEvent &event)
{
	// Dont need to do anything
}

//	===========================================================================
void CMenuWindow::handleMenuSelection(const CMenuEvent &event)
{
	// This can only have come from children..
	FREE_POINTER(m_subWindow);

	// Are we the root?
	if (m_parentMenuWindow == NULL)
	{
		this->closePopupWindow();
	}

	// Are we listened to?
	if (m_menuListener)
	{
		// This will destroy this window
		m_menuListener->handleMenuSelection(event);
	}
}

//	===========================================================================
void CMenuWindow::lauchSubWindow(CSubMenuMenuItem *item, IControl *control)
{
	// Create the position to have the window
	CPoint point;
	CWindowTools::getMousePosition(point);
	point.setXPosition(m_windowSize.getRight());
	point.setYPosition(control->getAbsoluteRect().getTop() + m_windowSize.getTop());

	FREE_POINTER(m_subWindow);
	CWindowAttributes attributes = *m_attributes;
	m_subWindow = new CMenuWindow(CRect(0, 0, 10, 10), m_attributes, item->getMutableSubMenu(), true, this);
	m_subWindow->setParentWindow(this->getParentWindow());
	m_subWindow->setParentMenuWindow(this);
	m_subWindow->registerMenuListener(this);
	m_subWindow->useIcons(m_useIcons);
	m_subWindow->setMenuColours(((CMenuRoot *)m_theControlRoot)->getMenuColours());
	m_subWindow->setFont(((CMenuRoot *)m_theControlRoot)->getFont());
	m_subWindow->openPopupWindow(point);
}

//	===========================================================================
void CMenuWindow::killSubWindow()
{
	FREE_POINTER(m_subWindow);
}

//	===========================================================================
void CMenuWindow::useIcons(const bool icons)
{
	m_useIcons = icons;
	((CMenuRoot *)m_theControlRoot)->useIcons(icons);
}	

//	===========================================================================
CMenuWindow::CMenuRoot::CMenuRoot(IWindow *parent) 
					  : CControlRoot(parent)
					  , m_menu(NULL)
					  , m_font(NULL)
					  , m_minimumWidth(0)
					  , m_currentSelection(0)
					  , m_guardSelection(0)
					  , m_useIcons(false)
{
	// Store the default colours
	m_colours.m_disabledTextColour   = CAlphaColour::CALPHACOLOUR_LIGHT_GREY;
	m_colours.m_frameColour		     = CAlphaColour::CALPHACOLOUR_PURPLE;
	m_colours.m_selectedColour	     = CAlphaColour::CALPHACOLOUR_PURPLE;
	m_colours.m_seperatorColour	     = CAlphaColour::CALPHACOLOUR_ORANGE;
	m_colours.m_subMenuPointerColour = CAlphaColour::CALPHACOLOUR_BLACK;
	m_colours.m_textColour			 = CAlphaColour::CALPHACOLOUR_BLACK;
	
	EXCHANGE_COUNTED_OBJECTS(m_font, CFont::CFONT_SYSTEM_FONT);

	// Set the selection
	this->setCurrentSelection();
}

//	===========================================================================
CMenuWindow::CMenuRoot::~CMenuRoot()
{
	FORGET_COUNTED_OBJECT(m_menu);
	FORGET_COUNTED_OBJECT(m_font);
}

//	===========================================================================
void CMenuWindow::CMenuRoot::drawRootControl(CGraphics &graphics)
{
	// Store the update region
	const CRect update = graphics.getUpdateArea();

	// Draw the window frame
	graphics.getMutablePen()->setColour(m_colours.m_frameColour);
	graphics.drawRectangle(CRect(0, 0, m_rootArea.getWidth(), m_rootArea.getHeight()));
	
	// Draw the selected rectangle
	if (m_currentSelection != CMENU_WINDOW_NO_SELECTION)
	{
		IControl *control = this->getControlAtIndex(m_currentSelection, true);

		if (control)
		{
			CRect selected = control->getArea();
			//selected.inset(2);
			graphics.getMutableBrush()->setColour(m_colours.m_selectedColour);
			graphics.fillRectangle(selected);
			graphics.drawRectangle(selected);
		}
	}

	// Now draw the menu items
	const CPoint drawingOffset = graphics.getDrawingAreaOffset();
	for (long i = 0; i < m_controlArray->getArraySize(); i++)
	{
		IControl *control = (IControl *)m_controlArray->elementAtIndex(i);
		if (control && update.rectanglesIntersect(control->getAbsoluteRect()))
		{
			// Setup a new clipping region
			graphics.getClippingRegion().setCurrentClipRegion(control->getAbsoluteRect());

			// Now reoffset the graphics area so that controls draw from 0, 0 relative to themselves
			graphics.setDrawingAreaOffset(control->getAbsoluteRect().getOrigin());

			// Draw 
			control->drawControl(graphics);

			// Now remove the offset from the graphics area so that future controls draw in the correct place....
			graphics.setDrawingAreaOffset(drawingOffset);

			// Reset the clipping region
			graphics.getClippingRegion().clearClipRegion();
		}
	}
	graphics.setDrawingAreaOffset(drawingOffset);
}

//	===========================================================================
void CMenuWindow::CMenuRoot::handleLeftButtonDown(CMouseEvent &event)
{
	m_guardSelection = m_currentSelection;
}

//	===========================================================================
void CMenuWindow::CMenuRoot::handleLeftButtonUp(CMouseEvent &event)
{
	if (m_currentSelection == m_guardSelection && m_currentSelection != CMENU_WINDOW_NO_SELECTION)
	{
		CMenuItemDisplay *theControl = (CMenuItemDisplay *)this->getControlAtIndex(m_currentSelection, true);

		// Check we got a valid control
		if (!theControl)
		{
			m_guardSelection = CMENU_WINDOW_NO_SELECTION;
			return;
		}

		// Get the menu item
		CMenuItem *item = theControl->getMutableMenuItem();

		// Check we got valid menu item
		if (!item)
		{
			m_guardSelection = CMENU_WINDOW_NO_SELECTION;
			return;
		}

		// Check if its inside the area
		if (!theControl->getAbsoluteRect().pointIsInside(event.getMousePosition()))
		{
			m_guardSelection = CMENU_WINDOW_NO_SELECTION;
			return;
		}

		// Now based on the item type we have to do something..
		switch(item->getType())
		{
			case CMenuItem::e_menuItemHtmlLink:
				{
					CHtmlMenuItem *myItem = (CHtmlMenuItem *)item;
					myItem->lauchLinkInBrowser();
					((CMenuWindow *)m_parentWindow)->handleMenuSelection(CMenuEvent(item, m_menu));
				}
			break;
			case CMenuItem::e_menuItemNormal:
					((CMenuWindow *)m_parentWindow)->handleMenuSelection(CMenuEvent(item, m_menu));
			break;
			case CMenuItem::e_menuItemSelectable:
				{
					CSelectableMenuItem *myItem = (CSelectableMenuItem *)item; 
					myItem->setSelected(!myItem->isSelected());
					((CMenuWindow *)m_parentWindow)->handleMenuSelection(CMenuEvent(item, m_menu));
				}
			break;
			case CMenuItem::e_menuItemSeperator:
			case CMenuItem::e_subMenuHolder:
			case CMenuItem::e_verticalSeperator:
				// Do nothing
			break;
		}
	}
}

//	===========================================================================
void CMenuWindow::CMenuRoot::handleMouseMovement(CMouseEvent &event)
{
	// Determine if we have a control
	IControl *theControl = this->preProcessMouseEvent(event);

	// If the control is null we dont need to go any further
	if (theControl == NULL)
	{
		return;
	}

	// Handle the mouse event
	CMenuItemDisplay *menuItem = (CMenuItemDisplay *)theControl;
	CMenuItem *item			   = menuItem->getMutableMenuItem();
	CMenuItemDisplay *last     = (CMenuItemDisplay *)this->getControlAtIndex(m_currentSelection, true);
	//const long lastIndex       = m_currentSelection;
	const long thisIndex       = this->getControlIndex(theControl);
	if (last)
	{
		last->update();
	}
	
	// Set the cursor
	if (item && item->getType() == CMenuItem::e_menuItemHtmlLink)
	{
		event.getMutableMouse()->setCursor(&CCursor::CCURSOR_HAND);
	}
	else
	{
		event.getMutableMouse()->setCursor();
	}

	// Now deal wih the menu item
	if (item && (m_currentSelection != thisIndex))
	{
		if (!item->isEnabled())
		{
			this->setCurrentSelection();
		}
		else
		{
			switch(item->getType())
			{
				case CMenuItem::e_menuItemNormal:
				case CMenuItem::e_menuItemSelectable:
				case CMenuItem::e_menuItemHtmlLink:
					this->setCurrentSelection(thisIndex);
					menuItem->update();
					/*if (last)
					{
						last->update();
					}*/
					((CMenuWindow *)m_parentWindow)->killSubWindow();
				break;
				case CMenuItem::e_menuItemSeperator:
					this->setCurrentSelection();
					menuItem->update();
					/*if (last)
					{
						last->update();
					}*/
					((CMenuWindow *)m_parentWindow)->killSubWindow();
				break;
				case CMenuItem::e_subMenuHolder:
					((CMenuWindow *)m_parentWindow)->preLauchSubWindow();
					this->setCurrentSelection(thisIndex);
					menuItem->update();
					/*if (last)
					{
						last->update();
					}*/
					((CMenuWindow *)m_parentWindow)->lauchSubWindow((CSubMenuMenuItem *)item, menuItem);
					((CMenuWindow *)m_parentWindow)->preLauchSubWindow(false);
				break;
			}
		}
	}

	// Process the changes
	this->postProcessMouseEvent(theControl);

	this->updateArea(m_rootArea);
}

//	===========================================================================
void CMenuWindow::CMenuRoot::handleMouseLeavingArea(CMouseEvent &event)
{
	if (m_currentSelection != CMENU_WINDOW_NO_SELECTION)
	{
		CMenuItem *item = m_menu->getMenuItem(m_currentSelection);
		if (item && item->getType() != CMenuItem::e_subMenuHolder)
		{
			CMenuItemDisplay *display = (CMenuItemDisplay *)this->getControlAtIndex(m_currentSelection);
			if (display)
			{
				display->mouseIsOver(false);
				display->update();
			}
			this->setCurrentSelection();
		}
	}
	this->updateArea(m_rootArea);
}

//	===========================================================================
void CMenuWindow::CMenuRoot::setMinimumWidth(const long width)
{
	m_minimumWidth = width;
}

//	===========================================================================
void CMenuWindow::CMenuRoot::findMaxStringDimension(CDimension &maxDimension)
{
	// On windows we require a draw context to compute the string size
#ifdef WIN32

	// Sme variables for GDI+
	bool hasGdiPlus		   = false;
	ULONG_PTR gdiPlusToken = 0;
	GdiplusStartupInput startupInput;

	// Start GDI+
	if (GdiplusStartup(&gdiPlusToken, &startupInput, NULL) == Ok)
	{
		hasGdiPlus = true;
	}

	HDC drawContext = GetDC(this->getParentWindow()->getWindowHandle()->m_windowHandle);
	HGDIOBJ old		= SelectObject(drawContext, m_font->getFont());
#endif

	// Rest the normal diemsion
	maxDimension.setWidth(0);
	maxDimension.setHeight(0);

	for (long i = 0; i < m_menu->getNumberOfMenuItems(); i++)
	{
		CMenuItem *item = m_menu->getMenuItem(i);
		if (item && item->getType() != CMenuItem::e_menuItemSeperator)
		{
			// Get the dimension of the string
			CDimension dimension;
			#ifdef WIN32
				if (hasGdiPlus)
				{
					CWindowTools::computeStringAreaNoGDIPlusStartup(drawContext, item->getText(), dimension, m_font->getFont());
				}
				else
				{
					CWindowTools::computeStringArea(drawContext, item->getText(), dimension, m_font->getFont());
				}
			#else
				CWindowTools::computeStringArea(m_font, item->getText(), dimension);
			#endif

			// Now store the width and height
			if (dimension.getWidth() > maxDimension.getWidth())
			{
#ifdef WIN32
				maxDimension.setWidth(dimension.getWidth());
#else
				maxDimension.setWidth(max(dimension.getWidth(), (long)TextWidth(item->getText().getString(), 0, strlen(item->getText().getString()))));
#endif
			}
			if (dimension.getHeight() > maxDimension.getHeight())
			{
				maxDimension.setHeight(dimension.getHeight());
			}
		}
	}

#ifdef WIN32
	if (hasGdiPlus)
	{
		GdiplusShutdown(gdiPlusToken);
	}
#endif
}

//	===========================================================================
bool CMenuWindow::CMenuRoot::findMaxIconDimension(CDimension &maxDimension)
{
	bool hasIcon = false;
	for (long i = 0; i < m_menu->getNumberOfMenuItems(); i++)
	{
		CMenuItem *item = m_menu->getMenuItem(i);
		if (item && item->getType() != CMenuItem::e_menuItemSeperator && item->getNormalImage() != NULL)
		{
			hasIcon = true;
			if (item->getNormalImage()->getImageSize().getWidth() > maxDimension.getWidth())
			{
				maxDimension.setWidth(item->getNormalImage()->getImageSize().getWidth());
			}
			if (item->getNormalImage()->getImageSize().getHeight() > maxDimension.getHeight())
			{
				maxDimension.setHeight(item->getNormalImage()->getImageSize().getHeight());
			}
		}
	}
	return hasIcon;
}

//	===========================================================================
bool CMenuWindow::CMenuRoot::doesMenuHaveExtraItems(bool &selectable)
{
	bool hasExtra = false;
	for (long i = 0; i < m_menu->getNumberOfMenuItems(); i++)
	{
		CMenuItem *item = m_menu->getMenuItem(i);
		if (item && item->getType() != CMenuItem::e_menuItemSeperator)
		{
			if (item->getType() == CMenuItem::e_menuItemSelectable)
			{
				selectable = true;
			}
			if (item->getType() == CMenuItem::e_subMenuHolder)
			{
				hasExtra = true;
			}
		}
	}
	return hasExtra;
}

//	===========================================================================
CDimension *CMenuWindow::CMenuRoot::setMenu(CMenu *menu)
{
	EXCHANGE_COUNTED_OBJECTS(m_menu, menu);
	if (m_menu)
	{
		// Menu item displays offset everything to have a 2 pixel border
		// The total height is the maximum height plus 4 (2 px border at the top)
		// The total width is the width of each item + 2

		CDimension maxIconDimension;
		CDimension maxStringDimension;

		bool select	  = false;
		bool hasSub   = this->doesMenuHaveExtraItems(select);
		bool hasIcons = this->findMaxIconDimension(maxIconDimension);
		this->findMaxStringDimension(maxStringDimension);

		// The height of each menu item is determined to be the biggest height of all available dimensions
		const long menuItemHeight = max(maxIconDimension.getHeight(), maxStringDimension.getHeight()) + 8;

		// Now the width is the combination of certain areas. First we have the label area
		long menuItemWidth = maxStringDimension.getWidth() + 8;

		// If we have sub menu items, we need an extra height added
		if (hasSub)
		{
			menuItemWidth += menuItemHeight;
		}

		// If we have icons, then we need to add their width on as well, 
		if (hasIcons)
		{
			menuItemWidth += maxIconDimension.getWidth();
		}
		else if (select)
		{
			menuItemWidth += menuItemHeight;
		}
		
		// Constrian to the min size as is necessary
		//if (menuItemHeight < 24)
		//{
		//	menuItemHeight = 24;
		//}
		if (menuItemWidth < m_minimumWidth)
		{
			menuItemWidth = m_minimumWidth;
		}

		// We want a 2px gap at the top and bottom
		//menuItemHeight += 4;
		//menuItemWidth  += 4;
		
		// Delete an previous controls
		this->clearControls();

		// Now we add the menu items
		long theTop  = 0;
		long theLeft = 0;

		for (long i = 0; i < m_menu->getNumberOfMenuItems(); i++)
		{
			CMenuItem *item = m_menu->getMenuItem(i);

			// If have an item
			if (item == NULL)
			{
				continue;
			}

			// Now handle based on the type
			if (item->getType() == CMenuItem::e_verticalSeperator)
			{
				theTop   = 0;
				theLeft += menuItemWidth + 4;
			}
			else
			{
				if (item->getType() == CMenuItem::e_menuItemSeperator)
				{
					CMenuItemDisplay *display = new CMenuItemDisplay(this, i, CRect(theLeft, theTop, menuItemWidth, 4), item, m_useIcons, select, hasSub, maxIconDimension.getWidth());
					display->setFont(m_font);
					display->setColours(m_colours);
					this->addControl(display);
					theTop += 4;
				}
				else
				{
					CMenuItemDisplay *display = new CMenuItemDisplay(this, i, CRect(theLeft, theTop, menuItemWidth, menuItemHeight), item, m_useIcons, select, hasSub, maxIconDimension.getWidth());
					display->setFont(m_font);
					display->setColours(m_colours);
					this->addControl(display);
					theTop += menuItemHeight;
				}
			}
		}

		// We hae built the menu now just return the size and we are done!
		return new CDimension(theLeft + menuItemWidth, theTop);
	}
	return NULL;
}

//	===========================================================================
void CMenuWindow::CMenuRoot::setFont(CFont *font)
{
	EXCHANGE_COUNTED_OBJECTS(m_font, font);
	const long numberOfItems = m_menu->getNumberOfMenuItems();
	for (long i = 0; i < numberOfItems; i++)
	{
		CMenuItemDisplay *item = (CMenuItemDisplay *)this->getControlAtIndex(i);
		if (item)
		{
			item->setFont(m_font);
		}
	}
}

//	===========================================================================
void CMenuWindow::CMenuRoot::setMenuColours(const SMenuColours &colours)
{
	// Store
	m_colours.m_disabledTextColour   = colours.m_disabledTextColour;
	m_colours.m_frameColour		     = colours.m_frameColour;
	m_colours.m_selectedColour	     = colours.m_selectedColour;
	m_colours.m_seperatorColour	     = colours.m_seperatorColour;
	m_colours.m_subMenuPointerColour = colours.m_subMenuPointerColour;
	m_colours.m_textColour			 = colours.m_textColour;

	// and pass on
	const long numberOfItems = m_menu->getNumberOfMenuItems();
	for (long i = 0; i < numberOfItems; i++)
	{
		CMenuItemDisplay *item = (CMenuItemDisplay *)this->getControlAtIndex(i);
		if (item)
		{
			item->setColours(m_colours);
		}
	}
}

//	===========================================================================
void CMenuWindow::CMenuRoot::setCurrentSelection(const long selection)
{
	if ((selection >= 0 && selection < m_menu->getNumberOfMenuItems()) || selection == CMENU_WINDOW_NO_SELECTION)
	{
		if (m_currentSelection != CMENU_WINDOW_NO_SELECTION)
		{
			CMenuItemDisplay *display = (CMenuItemDisplay *)this->getControlAtIndex(m_currentSelection);
			if (display)
			{
				display->mouseIsOver(false);
				display->update();
			}
		}

		m_currentSelection = selection;

		if (m_currentSelection != CMENU_WINDOW_NO_SELECTION)
		{
			CMenuItemDisplay *display = (CMenuItemDisplay *)this->getControlAtIndex(m_currentSelection);
			if (display)
			{
				display->mouseIsOver(true);
				display->update();
			}
		}
	}
}

//	===========================================================================
void CMenuWindow::CMenuRoot::useIcons(const bool icons)
{
	m_useIcons = icons;
	this->setMenu(m_menu);
}